<?php

namespace App\Http\Controllers\Admin\Menu;

use App\Http\Controllers\Controller;
use App\Models\Menu\Menu;
use App\Models\Menu\Item;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\DB;

class ItemController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
        $this->middleware('AdminHasPermission:menus');
    }

    public function index($menu_id)
    {
        $menu = Menu::findOrFail($menu_id);
        $items = Item::where('menu_id', $menu_id)
            ->orderBy('sort_order', 'asc')
            ->get();

        return view('Admin.Menu.Item.index', compact(
            'menu',
            'items'
        ));
    }

    public function create($menu_id)
    {
        $menu = Menu::findOrFail($menu_id);
        $parents = Item::where('menu_id', $menu_id)
            ->whereNull('parent_id')
            ->orderBy('sort_order', 'asc')
            ->get();

        return view('Admin.Menu.Item.create', compact(
            'menu',
            'parents'
        ));
    }

    public function store(Request $request)
    {
        $this->validate($request,[
            'menu_id' => 'required|numeric|exists:menus,id',
            'sort_order' => 'required|numeric',
            'title' => 'required|max:191',
            'link' => 'required',
            'parent_id' => 'nullable',
            'status' => 'required|numeric|in:0,1',
        ]);

        Item::create([
            'menu_id' => $request['menu_id'],
            'sort_order' => $request['sort_order'],
            'title' => $request['title'],
            'link' => $request['link'],
            'parent_id' => $request['parent_id'],
            'status' => $request['status']
        ]);

        flash_message('success', __('labels.alert_stored'), '');
        return redirect()->route('admin.menu.item.index', ['menu_id'=>$request['menu_id']]);
    }

    public function edit($id)
    {
        $item = Item::findOrFail($id);
        $parents = Item::where('menu_id', $item->menu_id)
            ->where('id', '<>', $id)
            ->whereNull('parent_id')
            ->orderBy('sort_order', 'asc')
            ->get();

        return view('Admin.Menu.Item.edit', compact(
            'item',
            'parents'
        ));
    }

    public function update(Request $request, $id)
    {
        $item = Item::findOrFail($id);
        $this->validate($request,[
            'sort_order' => 'required|numeric',
            'title' => 'required|max:191',
            'link' => 'required',
            'parent_id' => 'nullable',
            'status' => 'required|numeric|in:0,1',
        ]);

        $item->update([
            'sort_order' => $request['sort_order'],
            'title' => $request['title'],
            'link' => $request['link'],
            'parent_id' => $request['parent_id'],
            'status' => $request['status']
        ]);

        flash_message('success', __('labels.alert_stored'), '');
        return redirect()->route('admin.menu.item.index', ['menu_id'=>$item->menu_id]);
    }

    public function delete($id)
    {
        Item::findOrFail($id);
        Item::destroy($id);

        flash_message('info', __('labels.deleted_successfully'), '');
        return back();
    }

}




