<?php

namespace App\Http\Controllers\Admin\Blog;

use App\Http\Controllers\Controller;
use App\Models\Blog\ArticleCategory;
use App\Models\Blog\Article;
use Illuminate\Http\Request;

class ArticleController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
        $this->middleware('AdminHasPermission:blog');
    }

    public function index()
    {
        $items = Article::orderBy('created_at', 'desc')
            ->get();

        return view('Admin.Blog.Article.index', compact(
            'items'
        ));
    }

    public function create()
    {
        $categories = ArticleCategory::orderBy('sort_order', 'asc')
            ->get();

        return view('Admin.Blog.Article.create', compact(
            'categories'
        ));
    }

    public function store(Request $request)
    {
        $this->validate($request,[
            'article_category_id' => 'required|numeric',
            'title' => 'required|max:191',
            'slug' => 'required|max:191|unique:articles',
            'image' => 'nullable|image|mimes:jpg,jpeg,png|max:10000',
            'header_image' => 'nullable|image|mimes:jpg,jpeg,png|max:10000',
            'short_text' => 'nullable|max:1023',
            'status' => 'required|numeric|in:0,1'
        ],[
            'article_category_id.required' => 'دسته بندی را انتخاب کنید',
            'title.required' => 'عنوان را وارد کنید',
            'title.max' => 'کمتر از 191 کاراکتر وارد کنید',
            'slug.required' => 'نام دسترسی را وارد کنید',
            'slug.max' => 'کمتر از 191 کاراکتر وارد کنید',
            'slug.unique' => 'این نام دسترسی قبلا ثبت شده است',
            'image.image' => 'فایل انتخاب شده پشتیبانی نمیشود',
            'image.mimes' => 'تصویر انتخاب شده پشتیبانی نمیشود',
            'image.max' => 'حجم فایل کمتر از 10 مگ باشد',
            'header_image.image' => 'فایل انتخاب شده پشتیبانی نمیشود',
            'header_image.mimes' => 'تصویر انتخاب شده پشتیبانی نمیشود',
            'header_image.max' => 'حجم فایل کمتر از 10 مگ باشد',
            'short_text.max' => 'کمتر از 1023 کاراکتر وارد کنید',
            'status.required' => 'وضعیت را تعیین کنید'
        ]);

        $image = null;
        if(!empty($request['image'])){
            $sizes = getimagesize($request['image']);
            $image = store_image($request['image'], 'blog', $sizes[0], $sizes[1]);
        }
        $header_image = null;
        if(!empty($request['header_image'])){
            $sizes = getimagesize($request['header_image']);
            $header_image = store_image($request['header_image'], 'blog', $sizes[0], $sizes[1]);
        }

        Article::create([
            'article_category_id' => $request['article_category_id'],
            'title' => $request['title'],
            'slug' => $request['slug'],
            'image' => $image,
            'header_image' => $header_image,
            'short_text' => $request['short_text'],
            'text' => $request['text'],
            'status' => $request['status']
        ]);

        flash_message('success', 'با موفقیت ثبت شد', '');
        return redirect()->route('admin.blog.article.index');
    }

    public function edit($id)
    {
        $item = Article::findOrFail($id);
        $categories = ArticleCategory::orderBy('sort_order', 'asc')
            ->get();

        return view('Admin.Blog.Article.edit', compact(
            'item',
            'categories'
        ));
    }

    public function update(Request $request, $id)
    {
        $item = Article::findOrFail($id);

        $this->validate($request,[
            'article_category_id' => 'required|numeric',
            'title' => 'required|max:191',
            'slug' => 'required|max:191|unique:articles,slug,'.$item->id,
            'image' => 'nullable|image|mimes:jpg,jpeg,png|max:10000',
            'header_image' => 'nullable|image|mimes:jpg,jpeg,png|max:10000',
            'short_text' => 'nullable|max:1023',
            'status' => 'required|numeric|in:0,1'
        ],[
            'article_category_id.required' => 'دسته بندی را انتخاب کنید',
            'title.required' => 'عنوان را وارد کنید',
            'title.max' => 'کمتر از 191 کاراکتر وارد کنید',
            'slug.required' => 'نام دسترسی را وارد کنید',
            'slug.max' => 'کمتر از 191 کاراکتر وارد کنید',
            'slug.unique' => 'این نام دسترسی قبلا ثبت شده است',
            'image.image' => 'فایل انتخاب شده پشتیبانی نمیشود',
            'image.mimes' => 'تصویر انتخاب شده پشتیبانی نمیشود',
            'image.max' => 'حجم فایل کمتر از 10 مگ باشد',
            'header_image.image' => 'فایل انتخاب شده پشتیبانی نمیشود',
            'header_image.mimes' => 'تصویر انتخاب شده پشتیبانی نمیشود',
            'header_image.max' => 'حجم فایل کمتر از 10 مگ باشد',
            'short_text.max' => 'کمتر از 1023 کاراکتر وارد کنید',
            'status.required' => 'وضعیت را تعیین کنید'
        ]);

        $image = $item->image ?? null;
        if(!empty($request['image'])){
            try{
                unlink($item->image);
            } catch(\Exception $e){}

            $sizes = getimagesize($request['image']);
            $image = store_image($request['image'], 'blog', $sizes[0], $sizes[1]);
        }
        $header_image = $item->header_image ?? null;
        if(!empty($request['header_image'])){
            try{
                unlink($item->header_image);
            } catch(\Exception $e){}

            $sizes = getimagesize($request['header_image']);
            $header_image = store_image($request['header_image'], 'blog', $sizes[0], $sizes[1]);
        }

        $item->update([
            'article_category_id' => $request['article_category_id'],
            'title' => $request['title'],
            'slug' => $request['slug'],
            'image' => $image,
            'header_image' => $header_image,
            'short_text' => $request['short_text'],
            'text' => $request['text'],
            'status' => $request['status']
        ]);

        flash_message('success', 'با موفقیت ثبت شد', '');
        return redirect()->route('admin.blog.article.index');
    }

    public function delete($id)
    {
        $item = Article::findOrFail($id);

        try{
            unlink($item->image);
            unlink($item->header_image);
        } catch(\Exception $e){}

        Article::destroy($id);

        flash_message('info', 'حذف شد', '');
        return back();
    }

}




