<?php

namespace App\Http\Controllers\Admin\Admins;

use App\Models\Admin;
use App\Models\AdminAccessItem;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class AdminsController extends Controller
{
    public function __construct()
    {
        $this->middleware(['admin', 'Language']);
        $this->middleware('AdminHasPermission:admins');
    }

    public function list()
    {
        $admins = Admin::where('id', '<>', '1')
            ->where('is_delete', false)
            ->orderBy('id', 'desc')
            ->get();

        return view('Admin.Admins.list', compact('admins'));
    }

    public function create()
    {
        $access_items = AdminAccessItem::orderBy('id', 'desc')->get();

        return view('Admin.Admins.create', compact(
            'access_items'
        ));
    }

    public function store(Request $request)
    {
        $this->validate($request, [
            'email' => 'required|email|unique:admins',
            'mobile_number' => 'required|numeric|digits:11|unique:admins',
            'password' => 'required|min:6|max:191|confirmed'
        ]);

        $admin = new Admin();
        $admin = $admin->create([
            'email' => $request['email'],
            'mobile_number' => $request['mobile_number'],
            'password' => Hash::make($request['password'])
        ]);

        if(!empty($request['items'])){
            $admin->access()->attach($request['items']);
        }

        flash_message('success', __('labels.new_admin_stored'), '');
        return redirect()->route('admin.admins.list');
    }

    public function edit($id)
    {
        if($id == 1){
            abort(404);
        }

        $admin = Admin::findOrFail($id);
        if($admin->is_delete){
            abort(404);
        }

        $access_items = AdminAccessItem::orderBy('id', 'desc')->get();

        $selected_items = array();
        if(!empty($admin->access)){
            foreach($admin->access as $access){
                $selected_items[] = $access->id;
            }
        }

        return view('Admin.Admins.edit', compact(
            'admin',
            'selected_items',
            'access_items'
        ));
    }

    public function update(Request $request, $id)
    {
        if($id == 1){
            abort(404);
        }

        $admin = Admin::findOrFail($id);
        if($admin->is_delete){
            abort(404);
        }

        $this->validate($request, [
            'email' => 'required|email|unique:admins,email,'.$admin->id,
            'mobile_number' => 'required|numeric|digits:11|unique:admins,mobile_number,'.$admin->id,
            'password' => 'nullable|min:6|max:191|confirmed'
        ]);

        if(empty($request['password'])){
            $pass = $admin->password;
        }
        else{
            $pass = Hash::make($request['password']);
        }

        $admin->update([
            'email' => $request['email'],
            'mobile_number' => $request['mobile_number'],
            'password' => $pass
        ]);

        $admin->access()->sync($request['items']);

        flash_message('success', __('labels.updated'), '');
        return redirect()->route('admin.admins.list');
    }

    public function destroy($id)
    {
        if($id == '1'){
            abort(404);
        }
        $admin = Admin::findOrFail($id);
        if($admin->is_delete){
            abort(404);
        }

        $admin->update([
            'email' => $admin->email.'_deleted_'.$admin->id,
            'mobile_number' => $admin->mobile_number.'_deleted_'.$admin->id,
            'is_delete' => true
        ]);

        flash_message('info', __('labels.deleted'), '');
        return back();
    }
}
